/* 
testsisa.c	test operation of the SISA-1 board   version 0.1

		(C) S57UUU 10 jan 2021     GNU GPL v2

		loads firmware into the FPGA, then
		reads data from FT2232, bulk asynchronous,
		as packets:
		16 bit flag 0xAA55
		16 bit data length [16 bit words] meta+data
		16 bit meta1
		16 bit meta2
		16 bit meta3
		16 bit data
		16 bit data......

		length = meta and data only (without head,len)
		
		then draw FFT spectrums of the four channels
		
		
compile: 
gcc testsisa.c -o testsisa -lm -lusb-1.0 -lX11

data sent by sisa_4c.xise  (sisa_4c.bin)  on the FPGA


*/


#include <stdio.h>
#include <math.h>

#include <libusb-1.0/libusb.h>

#include <sys/resource.h>	/* setpriority */
#include <sched.h>		/* sched_setscheduler */
#include <sys/mman.h>		/* mlockall */


#define MAXTRANS 256
#define CHUNKSIZE 512
#define FFTSIZE 512

typedef unsigned char buffers[MAXTRANS][CHUNKSIZE];

typedef struct
  {
  double frekvenca,ddsclock;
  int ddsbits,phincr;
  unsigned char buf[10];	//for bulk out (set freq)
  int bytes;
  libusb_device_handle *handle;
  int konc;
  } ffstat;


#include "userdata.h"
#include "usbstuff.c"
#include "Xgraf.c"
#include "fftstuff.c"

  
//global  variables  
uint16_t paket[65536];	//packet data is copied here

float xr[FFTSIZE],xi[FFTSIZE],wr[FFTSIZE],wi[FFTSIZE],wind[FFTSIZE],yr[FFTSIZE],yi[FFTSIZE];
unsigned int bi[FFTSIZE];
int nfft=FFTSIZE;		//size of FFT
float maxr,maxi,maxa,maxamp,noise;
float xa[FFTSIZE],xla[FFTSIZE];
int bpp;

//---------------------------------------
//Adds contorl bits (clock, reset) to each bit in the bitstream
//file. Each bit becomes a byte, for simple sending to the FTDI chip.
//Returns the number of bytes created
int naredi_stream(unsigned char *buffer, int *n, char *filn)
{
unsigned char bitstream[2097152];
int i,j,bi,bit,nread,nbytes;
FILE *fin;

printf("Reading file %s\n",filn);
fin=fopen(filn,"r");
if (fin == NULL)
  {
  perror("Firmware");
  printf("Firmware file %s not found!\n");
  return -1;
  }

i=0;
do
  nread=fread(&bitstream[i++],1,1,fin);
while (nread == 1);
fclose(fin);
nbytes = i-1;
printf("Read %d bytes (%d bits)\n",nbytes,8*nbytes);

bi=0;
for (i=0;i<nbytes;i++)
  for (j=0;j<8;j++)
    {
    bit = (bitstream[i] >> (7-j)) & 0x01;
    buffer[bi++] = 0x00 + (bit << 3);	//data + clk dol
    buffer[bi++] = 0x04 + (bit << 3);	//data + clk gor
    }

printf("Loaded %d bytes\n",bi);

*n=bi;

return 0;
}

//-------------------------------------------------
//loads the FPGA firware. Sends the bytestream created by
//naredi_stream() via the FTDI chip.
//FT2232 must be in bitbang mode before calling this
int poslji_stream(libusb_device_handle *handle, int endpoint, unsigned char *buffer, int n, int timeout)
{
int i,er[16],bytes;
unsigned char obuf[16],ibuf[16];

//reset FPGA
obuf[0] = 0x00;	//INIT_B gor PROGRAM_B gor, ostali dol
er[9]=libusb_bulk_transfer(handle, endpoint, obuf, 1, &bytes, timeout);
obuf[0] = 0x80;	//INIT_B gor PROGRAM_B dol, ostali dol
er[9]=libusb_bulk_transfer(handle, endpoint, obuf, 1, &bytes, timeout);
obuf[0] = 0x00;	//INIT_B gor PROGRAM_B gor, ostali dol
er[9]=libusb_bulk_transfer(handle, endpoint, obuf, 1, &bytes, timeout);

usleep(5);

//Load configuration
er[9]=libusb_bulk_transfer(handle, endpoint, buffer, n, &bytes, timeout);
printf("Sent %d bytes\n",bytes);

return er[9];
}

//-----------------------------------------------
//draw decibel scale
void db_skala(int x, int y, int vis, float lminy, float lmaxy)
{
float ty;
int iy;
char s[256];

//lminy = 10.0*log10(miny);  lmaxy = 10.0*log10(maxy);

for (ty=lminy; ty<=lmaxy; ty+=10.0)
  {
  iy = y - ty/(lmaxy-lminy)*vis + 5;
  sprintf(s,"%3.0f",ty);
  XDrawString(dis,win,gc,x-28,iy,s,strlen(s));
  }
}

//-------------------------------------------------------------
//draw spectrum from complex FFT
//graf[]	input data
//tock		number of points
//miny,maxy	amplitude range
//x0.y0.sir.vis		position and size of graph
//cf = central frequency (bin)   from 0 to tock   (tock/2=dc)
//span = span (bins)   max tock   must be an even number!
void risigraf1(float graf[], long tock, float miny, float maxy, int x0, int y0, int sir, int vis, int cf, int span)
{
float x,y,sx,sy;
long j;
XPoint kriva[32768];
XSegment mreza[100];
int zac,kon,ind;

if (span>tock)
  {
  printf("Span too big (%d > %ld)\n",span,tock);
  return;
  }
if (abs(cf)>tock/2-1)
  {
  printf("Center frequency out of range (%d, max +-%ld\n",cf,tock/2-1);
  return;
  }
if (tock/2+cf-span/2<0)
  {
  printf("Low end (cf-span/2) out of range!\n");
  return;
  }
if (tock/2+cf+span/2>tock)
  {
  printf("High end (cf+span/2) out of range!\n");
  return;
  }

XSetForeground(dis,gc,barva[0]);		//brisi prejsnjo sliko
XFillRectangle(dis,win,gc,x0,y0,sir,vis+14);

XSetForeground(dis,gc,barva[5]);
j=0;
sy=vis/(maxy-miny)*10;
for (y=sy;y<vis*0.99;y=y+sy)
	{mreza[j].x1=x0;mreza[j].y1=y0+y;mreza[j].x2=x0+sir;mreza[j].y2=y0+y;j++;}
for (x=sir*0.1;x<=sir*0.91;x=x+sir*0.1)
	{mreza[j].x1=x0+x;mreza[j].y1=y0;mreza[j].x2=x0+x;mreza[j].y2=y0+vis;j++;}	
XDrawSegments(dis,win,gc,mreza,j);

XSetForeground(dis,gc,barva[8]);

kriva[0].x=x0;kriva[0].y=y0;	//okvir
kriva[1].x=x0+sir;kriva[1].y=y0;
kriva[2].x=x0+sir;kriva[2].y=y0+vis;
kriva[3].x=x0;kriva[3].y=y0+vis;
kriva[4].x=x0;kriva[4].y=y0;
XDrawLines(dis,win,gc,kriva,5,CoordModeOrigin);

XSetForeground(dis,gc,barva[15]);

zac=tock/2+cf-span/2;	//0=minf (index tock/2)  tock/2=DC  (index 0)  itd
kon=tock/2+cf+span/2;
sx=x0;
if (zac<tock/2) ind=zac+tock/2; else ind=zac-tock/2;
sy=y0+vis-(graf[ind]-miny)/(maxy-miny)*vis;
if (sy>(y0+vis)) sy=y0+vis; if (sy<y0) sy=y0;
for (j=zac;j<=kon;j++)
  {
  if (j<tock/2) ind=j+tock/2; else ind=j-tock/2;
  x=x0+(float)(j-zac)/(float)span*sir;
  y=y0+vis-(graf[ind]-miny)/(maxy-miny)*vis;
  if (y>(y0+vis)) y=y0+vis; if (y<y0) y=y0;
  kriva[j-zac].x=x; kriva[j-zac].y=y;
  }
XDrawLines(dis,win,gc,kriva,span+1,CoordModeOrigin);

db_skala(x0, y0, vis, miny, maxy);
}

//----------------------------------------------------
//separate data into four channels, calculate and draw FFT
void malaj_fft(uint16_t *paket)
{
static unsigned int presk,oldcnt;
int i,pktyp,cnt,dcnt;
float maxr,maxi,maxa,noise;
float xa[FFTSIZE],xla[FFTSIZE];
int x,y;
//float maxamp=32678.0*nfft/sqrtf(2.0);
float maxamp=32678.0*nfft*sqrtf(2.0);
char string[255];
int max,min;
int smpr,smpi;
float dcr,dci;
float avg,faza,amp1,amp2,amp3,amp4;

//check for lost packets
//when starting, it will always report some lost packets
pktyp = paket[0] >> 12;
cnt = paket[0] & 0x0FFF;
dcnt = (cnt-oldcnt)&0x0FFF;
oldcnt=cnt;
if (dcnt > 1)  printf("%d lost packets!\n",dcnt-1);

if (presk >= 100)	//skip packets (slow down drawing)
  {
  presk=0;

//---------  channel 0 
  maxr=-1E20; maxi=-1E20; max=-1000000; min=1000000;
  dcr=0;dci=0;amp1=0;
  for (i=0;i<8*nfft;i+=8)
    {
    smpr = paket[i+4];
    smpr = smpr>32767 ? smpr-65536 : smpr;
    if (smpr>max) max=smpr;
    if (smpr<min) min=smpr;
    dcr+=smpr;
    xr[i/8] = (float)smpr * wind[i/8];
    smpi = paket[i+3];
    smpi = smpi>32767 ? smpi-65536 : smpi;
    if (smpi>max) max=smpi;
    if (smpi<min) min=smpi;
    dci+=smpi;
    xi[i/8] = (float)smpi * wind[i/8];
    if (fabsf(xr[i/8])>maxr) maxr=fabsf(xr[i/8]);
    if (fabsf(xi[i/8])>maxi) maxi=fabsf(xi[i/8]);
    amp1 += hypotf(smpi,smpr);
//    dcr+=xr[i/8]; dci+=xi[i/8];
    }
  dcr/=nfft; dci/=nfft; amp1/=nfft;
//  printf("max=%6d (%04X)    min=%6d (%04X)       dcr=%6.1f  dci=%6.1f\n",max,(uint16_t)max,min,(uint16_t)min,dcr,dci);

  fft_f(nfft, xr, xi, wr, wi, bi);
   
  avg=0.0;
  for (i=0;i<nfft;i++)
    {
    xa[i]=hypotf(xi[i],xr[i])*1.3;
//    if (i==0) xa[i]=1E-20;	//odstrani DC peak
    avg+=xa[i]*xa[i];
    if (xa[i]>0) xla[i]=20.0*log10(xa[i]/maxamp); else xla[i]=-300;
    if (xa[i]>maxa) maxa=xa[i];
    }
  avg/=(float)(nfft-1);
//  printf("avg=%f ",avg);
    
  //draw graph
  x = 50;
  y = 30;
  risigraf1(xla, nfft, -120.0, 0.0, x, y, 410, 260, 0, nfft); 
  
//---------  channel 1
  maxr=-1E20; maxi=-1E20; max=-1000000; min=1000000;
  dcr=0;dci=0;amp2=0;
  for (i=0;i<8*nfft;i+=8)
    {
    smpr = paket[i+6];
    smpr = smpr>32767 ? smpr-65536 : smpr;
    if (smpr>max) max=smpr;
    if (smpr<min) min=smpr;
    dcr+=smpr;
    yr[i/8] = (float)smpr * wind[i/8];
    smpi = paket[i+5];
    smpi = smpi>32767 ? smpi-65536 : smpi;
    if (smpi>max) max=smpi;
    if (smpi<min) min=smpi;
    dci+=smpi;
    yi[i/8] = (float)smpi * wind[i/8];
    if (fabsf(yr[i/8])>maxr) maxr=fabsf(yr[i/8]);
    if (fabsf(yi[i/8])>maxi) maxi=fabsf(yi[i/8]);
    amp2+=hypotf(smpi,smpr);
//    dcr+=xr[i/8]; dci+=xi[i/8];
    }
  dcr/=nfft; dci/=nfft; amp2/=nfft;
//  printf("max=%6d (%04X)    min=%6d (%04X)       dcr=%6.1f  dci=%6.1f\n",max,(uint16_t)max,min,(uint16_t)min,dcr,dci);

  fft_f(nfft, yr, yi, wr, wi, bi);
    
  avg=0.0;
  for (i=0;i<nfft;i++)
    {
    xa[i]=hypotf(yi[i],yr[i])*1.3;
//    if (i==0) xa[i]=1E-20;	//odstrani DC peak
    avg+=xa[i]*xa[i];
    if (xa[i]>0) xla[i]=20.0*log10(xa[i]/maxamp); else xla[i]=-300;
    if (xa[i]>maxa) maxa=xa[i];
    }
  avg/=(float)(nfft-1);
//  printf("avg=%f ",avg);
    
  //draw graph
  x = 500;
  y = 30;
  risigraf1(xla, nfft, -120.0, 0.0, x, y, 410, 260, 0, nfft); 
  XFlush(dis);

//---------  channel 2
  maxr=-1E20; maxi=-1E20; max=-1000000; min=1000000;
  dcr=0;dci=0;amp3=0;
  for (i=0;i<8*nfft;i+=8)
    {
    smpr = paket[i+8];
    smpr = smpr>32767 ? smpr-65536 : smpr;
    if (smpr>max) max=smpr;
    if (smpr<min) min=smpr;
    dcr+=smpr;
    xr[i/8] = (float)smpr * wind[i/8];
    smpi = paket[i+7];
    smpi = smpi>32767 ? smpi-65536 : smpi;
    if (smpi>max) max=smpi;
    if (smpi<min) min=smpi;
    dci+=smpi;
    xi[i/8] = (float)smpi * wind[i/8];
    if (fabsf(xr[i/8])>maxr) maxr=fabsf(xr[i/8]);
    if (fabsf(xi[i/8])>maxi) maxi=fabsf(xi[i/8]);
    amp3 += hypotf(smpi,smpr);
//    dcr+=xr[i/8]; dci+=xi[i/8];
    }
  dcr/=nfft; dci/=nfft; amp3/=nfft;
//  printf("max=%6d (%04X)    min=%6d (%04X)       dcr=%6.1f  dci=%6.1f\n",max,(uint16_t)max,min,(uint16_t)min,dcr,dci);

  fft_f(nfft, xr, xi, wr, wi, bi);
    
  avg=0.0;
  for (i=0;i<nfft;i++)
    {
    xa[i]=hypotf(xi[i],xr[i])*1.3;
//    if (i==0) xa[i]=1E-20;	//odstrani DC peak
    avg+=xa[i]*xa[i];
    if (xa[i]>0) xla[i]=20.0*log10(xa[i]/maxamp); else xla[i]=-300;
    if (xa[i]>maxa) maxa=xa[i];
    }
  avg/=(float)(nfft-1);
//  printf("avg=%f ",avg);
    
  //draw graph
  x = 50;
  y = 350;
  risigraf1(xla, nfft, -120.0, 0.0, x, y, 410, 260, 0, nfft); 

//---------  channel 3
  maxr=-1E20; maxi=-1E20; max=-1000000; min=1000000;
  dcr=0;dci=0;amp4=0;
  for (i=0;i<8*nfft;i+=8)
    {
    smpr = paket[i+10];
    smpr = smpr>32767 ? smpr-65536 : smpr;
    if (smpr>max) max=smpr;
    if (smpr<min) min=smpr;
    dcr+=smpr;
    yr[i/8] = (float)smpr * wind[i/8];
    smpi = paket[i+9];
    smpi = smpi>32767 ? smpi-65536 : smpi;
    if (smpi>max) max=smpi;
    if (smpi<min) min=smpi;
    dci+=smpi;
    yi[i/8] = (float)smpi * wind[i/8];
    if (fabsf(yr[i/8])>maxr) maxr=fabsf(yr[i/8]);
    if (fabsf(yi[i/8])>maxi) maxi=fabsf(yi[i/8]);
    amp4+=hypotf(smpi,smpr);
//    dcr+=xr[i/8]; dci+=xi[i/8];
    }
  dcr/=nfft; dci/=nfft; amp4/=nfft;
//  printf("max=%6d (%04X)    min=%6d (%04X)       dcr=%6.1f  dci=%6.1f\n",max,(uint16_t)max,min,(uint16_t)min,dcr,dci);

  fft_f(nfft, yr, yi, wr, wi, bi);
    
  avg=0.0;
  for (i=0;i<nfft;i++)
    {
    xa[i]=hypotf(yi[i],yr[i])*1.3;
//    if (i==0) xa[i]=1E-20;	//odstrani DC peak
    avg+=xa[i]*xa[i];
    if (xa[i]>0) xla[i]=20.0*log10(xa[i]/maxamp); else xla[i]=-300;
    if (xa[i]>maxa) maxa=xa[i];
    }
  avg/=(float)(nfft-1);
//  printf("avg=%f ",avg);
    
  //draw graph
  x = 500;
  y = 350;
  risigraf1(xla, nfft, -120.0, 0.0, x, y, 410, 260, 0, nfft); 
  XFlush(dis);
  

  XSetForeground(dis,gc,barva[15]);
  sprintf(string,"Arrow keys:  left/right= +- 10kHz   up/down= +-100kHz    pgup/pgdn= +- 1MHz     Q=quit");
  XDrawString(dis,win,gc,50,660,string,strlen(string));
  
  XSetForeground(dis,gc,barva[0]);	//brisi prejsnji
  XFillRectangle(dis,win,gc,647,650,130,10);
  XSetForeground(dis,gc,barva[15]);
  sprintf(string,"META = %04X %04X %04X",paket[0],paket[1],paket[2]);
  XDrawString(dis,win,gc,650,660,string,strlen(string));

  }
else
  presk++;
//printf("%d ",presk);
}

//-----------------------------------------------------
//draw frequencies
void risi_frek(ffstat *stat)
{
int x,y;
char string[256];

//izpis frekvence
x=50; y=30;
XSetForeground(dis,gc,barva[0]);	//brisi prejsnji
XFillRectangle(dis,win,gc,x+170,y+275,100,10);
XFillRectangle(dis,win,gc,x+620,y+275,100,10);
XFillRectangle(dis,win,gc,x+170,y+595,100,10);
XFillRectangle(dis,win,gc,x+620,y+595,100,10);
XSetForeground(dis,gc,barva[15]);
sprintf(string," %7.4lf MHz",stat->frekvenca/1E6);
XDrawString(dis,win,gc,x+208-3.3*strlen(string),y+285,string,strlen(string));
XDrawString(dis,win,gc,x+658-3.3*strlen(string),y+285,string,strlen(string));
XDrawString(dis,win,gc,x+208-3.3*strlen(string),y+605,string,strlen(string));
XDrawString(dis,win,gc,x+658-3.3*strlen(string),y+605,string,strlen(string));
sprintf(string,"-500k",stat->frekvenca/1E6);
XDrawString(dis,win,gc,x-10,y+285,string,strlen(string));
XDrawString(dis,win,gc,x+440,y+285,string,strlen(string));
XDrawString(dis,win,gc,x-10,y+605,string,strlen(string));
XDrawString(dis,win,gc,x+440,y+605,string,strlen(string));
sprintf(string,"+500k",stat->frekvenca/1E6);
XDrawString(dis,win,gc,x+390,y+285,string,strlen(string));
XDrawString(dis,win,gc,x+840,y+285,string,strlen(string));
XDrawString(dis,win,gc,x+390,y+605,string,strlen(string));
XDrawString(dis,win,gc,x+840,y+605,string,strlen(string));
  
}

//----------------------------------------------------
//called for each received USB packet (typ 512 bytes or less)
//reads words, MSB first
//waits for header, reads number of data words, then data
//when a full SISA packet (typically 4099 words, 3 meta + 4096 data)
//is assembled, calls user functions to process the data
void unpack(unsigned char *buffer, int bytes, ffstat *stat)
{
int i;
static unsigned int stanje,dolzina,pkp,dpk;
static unsigned int j;

//printf("%d bytes\n",bytes);  
i=2;	//first two bytes in packet are garbage (FT2232 kink....?)
while (i<bytes)
  {
  switch (stanje)	//state machine for packet extraction
    {
    case 0:		//wait for 0xAA
//      printf("0");
      if (buffer[i]==0xAA) stanje=1; //else printf("H");
      break;
    case 1:		//wait for 0x55
//      printf("1");
      if (buffer[i]==0x55) stanje=2; else stanje=0;
      break;
    case 2:		//first byte (MSB) of packet length
//      printf("2");
      dolzina = 256*buffer[i];
      stanje=3;
      break;
    case 3:		//second byte (LSB) of packet length
//      printf("3");
      dolzina += buffer[i];
//      printf("d=%d ",dolzina);
      dpk=dolzina;
      pkp=0;
      stanje=4;
      break;
    case 4:		//first data byte (MSB)
//      printf("4");
      paket[pkp]=256*buffer[i];
      stanje=5;
      break;
    case 5:		//secnd data byte(LSB)
//      printf("5");
//      printf("5=%d ",dolzina);
      paket[pkp]+=buffer[i];
      pkp++;
      dolzina--;
      if (dolzina==0)	//SISA packet ready, do processing
	{
	//call user functions to process data
	malaj_fft(paket);
	risi_frek(stat);
	
	stanje=0;
	}
      else stanje=4;
      break;
    default: stanje=0; pkp=0; break;
    }
    i++;
  }
}

//------------------------------------------------------
//this one is called by "libusb_handle_events" after a transfer
//is completed
static void callback(struct libusb_transfer *trans)
{
cbdata *ud;
//int i
int er;

ud=(cbdata*)trans->user_data;
//printf("Callback %d ",ud->index);
//*ud->which=ud->index;

if (trans->status != LIBUSB_TRANSFER_COMPLETED)
	{
//	printf("stat=%d   ",trans->status);
//	perror("Transfer error");fflush(stdout);
	transfer_error(trans);
	}

unpack(trans->buffer, trans->actual_length, ud->stat);

er=libusb_submit_transfer(trans);

}

//---------------------------------------------------
//set the DDS inside FPGA to stat->frekvenca Hz
//commands to FPGA are 6 byte strings, first byte is which command
//command 1 is set frequency all channels: 01 FF FF FF xx xx
//FF FF FF is a 24 bit phase increment for the DDS,  xx = don't care
//since this is currently the only command implemented, first
//byte is not really checked in the FPGA
void set_freq(ffstat *stat)
{
stat->phincr = stat->frekvenca/stat->ddsclock*pow(2.0,stat->ddsbits);
stat->buf[0] = 1;	//commnad 1=set freq all chans
stat->buf[1] = (stat->phincr >> 16) & 0xFF;
stat->buf[2] = (stat->phincr >> 8) & 0xFF;
stat->buf[3] = stat->phincr & 0xFF;
libusb_bulk_transfer(stat->handle, 2, stat->buf, 4, &stat->bytes, 1000);
}

//-----------------------------------------------
//process key press event  (user interface)
void tipka(ffstat *stat, XEvent event)
{
KeySym key;
char ch,string[256];
int i,nc;

nc=XLookupString(&event.xkey,string,255,&key,0);
if (nc==1)		//"printable" key
  {
  if (string[0]==27)	//escape, exit
    {
    printf("\n\n");fflush(stdout);
    stat->konc=1;
    }
  ch=string[0];
  switch(ch)  //printable key
    {
    case 'q':		//Q = quit
    case 'Q':
      stat->konc=1;
      break;
    }
  }	//if printable key
else	//arrows, function keys
  {
  ch=0;
  i=(i+1)%11;
//  printf(" Nc=%d, key= %d \n",nc,key);fflush(stdout);
  switch(key)  //non-printable tipke
    {
    case 65361:	//left
      {
      stat->frekvenca -= 10000;
      if (stat->frekvenca < 0) stat->frekvenca = 0;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    case 65363:	//right
      {
      stat->frekvenca += 10000;
      if (stat->frekvenca > 32E6) stat->frekvenca = 32E6;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    case 65362:	//up
      {
      stat->frekvenca += 100000;
      if (stat->frekvenca > 32E6) stat->frekvenca = 32E6;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    case 65364:	//down
      {
      stat->frekvenca -= 100000;
      if (stat->frekvenca < 0) stat->frekvenca = 0;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    case 65365:	//pg up
      {
      stat->frekvenca += 1000000;
      if (stat->frekvenca > 32E6) stat->frekvenca = 32E6;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    case 65366:	//pg dn
      {
      stat->frekvenca -= 1000000;
      if (stat->frekvenca < 0) stat->frekvenca = 0;
//      printf("Frequency = %f\n",stat->frekvenca);
      set_freq(stat);
      break;
      }
    default:
      {
      break;
      }
    }
  }	//if printable key else
}

//****************************************************************
int main()
{
libusb_device_handle *handle,*handle2;
libusb_context *context,*context2;
int nt=MAXTRANS;	//number of simultaneous transfers, max = 128
struct libusb_transfer *trans[MAXTRANS];
buffers buff;
cbdata ud[MAXTRANS];
int er[16],endpoint2out;
int i,n;
int timeout=1000;
unsigned char buffer[16777216];	//for FPGA configuration byte stream
ffstat stat;
XEvent event;

printf("sisa_fft_4x16c starting\n");

fftmojprip_f(nfft, wr, wi, bi);	//prepare tables for FFT
pripwin(nfft, wind, 4, 1.0);	//prepare window for FFT

init_x(1000,690,"sisa_fft_4x16c_UI","TESTSISA");
setpal16(barva);
bpp=DefaultDepth(dis,screen);

set_priority();	///this will fail if not superuser, don't worry

//load firmware into FPGA
printf("Loading firmware\n");
er[2]=ft2232_get_handle(&handle2, &context2, 1);
er[3]=ft2232_set_bitbang_a(handle2, 2, 0xAC);
naredi_stream(buffer, &n, "sisa_4c.bin");
endpoint2out=4;
er[3]=poslji_stream(handle2, endpoint2out, buffer, n, timeout);
sleep(1);
libusb_close(handle2);


er[1]=ft2232_get_handle(&handle, &context, 0);
ft2232_set_syncfifo(handle);	//put FTDI into FIFO mode

stat.konc=0;
stat.handle = handle;

for (i=0;i<nt;i++)
  {
  ud[i].index=i;
  ud[i].stat=&stat;
  }
  
fill_transfers(handle, &callback, buff, ud, trans, timeout);

printf("Submitting transfers...\n");
for (i=0;i<nt;i++)
  er[9]=libusb_submit_transfer(trans[i]);

for (i=0;i<8;i++) stat.buf[i]=0;

stat.ddsclock = 64.0E6;		//master clock on SISA board
stat.ddsbits = 24;		//DDS frequency resolution
stat.frekvenca = 10.0E6;	//center frequency at start
set_freq(&stat);
//for (i=0;i<8;i++) printf("%02X  ",stat.buf[i]);printf("\n");

//main loop
while (stat.konc == 0)
  {
  er[10]=libusb_handle_events(context);

  if (XCheckMaskEvent(dis, ExposureMask|ButtonPressMask|KeyPressMask, &event))
    switch (event.type)
      {
      case KeyPress:
	tipka(&stat,event);
	break;
      default: break;
      }  
  }

}


