`timescale 1ns / 1ps
//////////////////////////////////////////////////////////////////////////////////
//
//	sisa_4c	   four channel sampler & down converter, clk=srate=64MHz
//					fixed decimation by 64 (output sample rate 1MHz complex)
//					Total data rate over USB < 130Mb/s  (< 50% USB2 capacity)
//
//					(C) S57UUU 2019   under GNU GPL v2
//
//		input signal pipeline:	
//						1. multiply  (downconvert mixing)
//						2. CIC LP filter & 32x decimate
//						3. FIR LP filter & 2x decimate
//						4. short FIFO
//						5. Multiplexer & packer
//						6. long FIFO
//						7. send to USB chip
//
//		items 1...4 are implemented twice for each channel (I and Q)
//
//		DDC LO is a 24 bit DDS, common to all channels, controlled over USB
//		with a 4 byte command "01 XX XX XX". XX XX XX ia a 24 bit phase increment
//
//		packet format:
//						0xAA55		packed header (fixed number)
//						LLLL			16 bit number of data words
//						META1			16 bit metadata 1		(packet type & counter)
//						META2			16 bit metadata 2		(currently fixed 0x2222)	
//						META3			16 bit metadata 3		(currently fixed 0x3333)
//						DATA			16 bit data word 0	(first sample chan 0 I)
//						DATA			16 bit data word 1	(first sample chan 0 Q)
//						DATA			16 bit data word 2	(first sample chan 1 I)
//						DATA			16 bit data word 3	(first sample chan 1 Q)
//						DATA			16 bit data word 4	(first sample chan 2 I)
//						DATA			16 bit data word 5	(first sample chan 2 Q)
//						DATA			16 bit data word 6	(first sample chan 3 I)
//						DATA			16 bit data word 7	(first sample chan 3 Q)
//						DATA			16 bit data word 8	(second sample chan 0 I)
//						........
//						DATA			16 bit data word LLLL-4
//
//		currently, LLLL=4096, 512 complex samples for each channel
//
//////////////////////////////////////////////////////////////////////////////////
module sisa_4c(
    input clk,
	 
	 input [11:0] adc1,
	 input [11:0] adc2,
	 input [11:0] adc3,
	 input [11:0] adc4,
	 output adc1clk,
	 output adc2clk,
	 output adc3clk,
	 output adc4clk,
	 
    input ftdi_clk,
	 input ftdi_txe_n,
	 input ftdi_rxf_n,
    output ftdi_rd_n,
    output ftdi_wr_n,
    output ftdi_siwu,
    output ftdi_oe_n,
    inout [7:0] ftdi_data,
	 
	 output zmig1
//	 output zmig2
	 
    );

reg [15:0] doutr,ad_data1,ad_data2,ad_data3,ad_data4;
wire [15:0]dout;
wire pk_data_av;
wire full_1;
wire full_2;
wire overflow;
wire rd_en;
wire signed [13:0] LOsine, LOcosine;
wire signed [15:0] sine_t, cosine_t;
wire signed [16:0] re1, im1, re2, im2, re3, im3, re4, im4;
wire signed [16:0] re1_f1, im1_f1, re2_f1, im2_f1, re3_f1, im3_f1, re4_f1, im4_f1;
wire signed [23:0] re1_f2, im1_f2, re2_f2, im2_f2, re3_f2, im3_f2, re4_f2, im4_f2;
wire signed [15:0] re1_ff, im1_ff, re2_ff, im2_ff, re3_ff, im3_ff, re4_ff, im4_ff;

wire [7:0] fifo_out, fifoempty;
reg [7:0]  ftdi_data_r;
reg [7:0]  ftdi_data_in_r;
reg ftdi_wr_n_r, fiforead;
reg ftdi_rd_n_r, ftdi_oe_n_r;
reg [47:0] commandword;
reg commandstrobe;

//wire rst = ~rst_n; // make reset active high
wire rst = 1'b0; // no reset input
 
assign ftdi_data = ftdi_data_r;
assign ftdi_wr_n = ftdi_wr_n_r;
assign ftdi_rd_n = ftdi_rd_n_r;
assign ftdi_oe_n = ftdi_oe_n_r;

//assign ftdi_oe_n = 1'b1;		//FTDI input only 
//assign ftdi_rd_n = 1'b1;
assign ftdi_siwu = 1'b1;		//must be H, when not in use!

assign adc1clk = clk;
assign adc2clk = clk;
assign adc3clk = clk;
assign adc4clk = clk;

wire [3 : 0] pktyp = 3;	//packet type	4 chans 16bit complex

always @(posedge clk)
	begin
	doutr <= dout;
	ad_data1 <= {adc1, 4'b0};
	ad_data2 <= {adc2, 4'b0};
	ad_data3 <= {adc3, 4'b0};
	ad_data4 <= {adc4, 4'b0};
	end;
/*
//test signal generator (for debugging only)
dds_t test_gen (
  .clk(clk), // input clk
  .cosine(cosine_t), // output [13 : 0] cosine
  .sine(sine_t) // output [13 : 0] sine
);
*/
//generator LO
dds1 dds15M (
  .clk(clk), // input clk
  .we(commandstrobe), // input we
  .data(commandword[39:16]), // input [24 : 0] phase incr
  .cosine(LOcosine), // output [13 : 0] cosine
  .sine(LOsine) // output [13 : 0] sine
);

//   ******** MIXING
mul1 mul_re1 (
  .clk(clk), // input clk
  .a(ad_data1), // input [15 : 0] a	
//  .a({cosine_t,2'b0}),	//TEST!!!
//  .a(16'b0),	//TEST!!!
  .b(LOcosine), // LO input [13 : 0] b
  .p(re1) // output [15 : 0] p
);

mul1 mul_im1 (
  .clk(clk), // input clk
  .a(ad_data1), // input [15 : 0] a 
//  .a({cosine_t,2'b0}),	//TEST!!!
//  .a(16'b0),	//TEST!!!
  .b(LOsine), // input [13 : 0] b
  .p(im1) // output [15 : 0] p
);

mul1 mul_re2 (
  .clk(clk), // input clk
  .a(ad_data2), // input [15 : 0] a	
//  .a({cosine_t,2'b0}),	//TEST!!!
//  .a(16'b0),	//TEST!!!
  .b(LOcosine), // input [13 : 0] b
  .p(re2) // output [15 : 0] p
);

mul1 mul_im2 (
  .clk(clk), // input clk
  .a(ad_data2), // input [15 : 0] a 
//  .a({cosine_t,2'b0}),	//TEST!!!
 // .a(16'b0),	//TEST!!!
  .b(LOsine), // input [13 : 0] b
  .p(im2) // output [15 : 0] p
);

mul1 mul_re3 (
  .clk(clk), // input clk
  .a(ad_data3), // input [15 : 0] a	
//  .a({cosine_t,2'b0}),	//TEST!!!
  .b(LOcosine), // input [13 : 0] b
  .p(re3) // output [15 : 0] p
);

mul1 mul_im3 (
  .clk(clk), // input clk
  .a(ad_data3), // input [15 : 0] a 
//  .a({cosine_t,2'b0}),	//TEST!!!
  .b(LOsine), // input [13 : 0] b
  .p(im3) // output [15 : 0] p
);

mul1 mul_re4 (
  .clk(clk), // input clk
  .a(ad_data4), // input [15 : 0] a	
//  .a({cosine_t,2'b0}),	//TEST!!!
  .b(LOcosine), // input [13 : 0] b
  .p(re4) // output [15 : 0] p
);

mul1 mul_im4 (
  .clk(clk), // input clk
  .a(ad_data4), // input [15 : 0] a 
//  .a({cosine_t,2'b0}),	//TEST!!!
  .b(LOsine), // input [13 : 0] b
  .p(im4) // output [15 : 0] p
);


//    ******** CIC decimation
cic1 cic_re1 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(re1), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h100), // TEST!!!!   ven pride 00FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready1_1), // output s_axis_data_tready
  .m_axis_data_tdata(re1_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_re1) // output m_axis_data_tvalid
);

cic1 cic_im1 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(im1), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h200), // TEST!!!!   ven pride 01FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready1_2), // output s_axis_data_tready
  .m_axis_data_tdata(im1_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_im1) // output m_axis_data_tvalid
);

cic1 cic_re2 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(re2), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h300), // TEST!!!!   ven pride 02FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready2_1), // output s_axis_data_tready
  .m_axis_data_tdata(re2_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_re2) // output m_axis_data_tvalid
);

cic1 cic_im2 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(im2), // input [15 : 0] s_axis_data_tdata
//   .s_axis_data_tdata(16'h400), // TEST!!!!   ven pride 03FF
 .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready2_2), // output s_axis_data_tready
  .m_axis_data_tdata(im2_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_im2) // output m_axis_data_tvalid
);

cic1 cic_re3 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(re3), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h500), // TEST!!!!   ven pride 04FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready3_1), // output s_axis_data_tready
  .m_axis_data_tdata(re3_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_re3) // output m_axis_data_tvalid
);

cic1 cic_im3 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(im3), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h600), // TEST!!!!   ven pride 05FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready3_2), // output s_axis_data_tready
  .m_axis_data_tdata(im3_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_im3) // output m_axis_data_tvalid
);

cic1 cic_re4 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(re4), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h700), // TEST!!!!   ven pride 06FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready4_1), // output s_axis_data_tready
  .m_axis_data_tdata(re4_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_re4) // output m_axis_data_tvalid
);

cic1 cic_im4 (
  .aclk(clk), // input aclk
  .s_axis_data_tdata(im4), // input [15 : 0] s_axis_data_tdata
//  .s_axis_data_tdata(16'h800), // TEST!!!!   ven pride 07FF
  .s_axis_data_tvalid(1'b1), // input s_axis_data_tvalid
  .s_axis_data_tready(ready4_2), // output s_axis_data_tready
  .m_axis_data_tdata(im4_f1), // output [15 : 0] m_axis_data_tdata
  .m_axis_data_tvalid(valid_im4) // output m_axis_data_tvalid
);

//    ******** FIR decimation
//  +2 compensates the DC offset from CIC
fir fir_re1 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_re1), // input s_axis_data_tvalid
  .s_axis_data_tready(tready1), // output s_axis_data_tready
  .s_axis_data_tdata(re1_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_re1), // output m_axis_data_tvalid
  .m_axis_data_tdata(re1_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_im1 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_im1), // input s_axis_data_tvalid
  .s_axis_data_tready(tready2), // output s_axis_data_tready
  .s_axis_data_tdata(im1_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_im1), // output m_axis_data_tvalid
  .m_axis_data_tdata(im1_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_re2 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_re2), // input s_axis_data_tvalid
  .s_axis_data_tready(tready1), // output s_axis_data_tready
  .s_axis_data_tdata(re2_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_re2), // output m_axis_data_tvalid
  .m_axis_data_tdata(re2_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_im2 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_im2), // input s_axis_data_tvalid
  .s_axis_data_tready(tready2), // output s_axis_data_tready
  .s_axis_data_tdata(im2_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_im2), // output m_axis_data_tvalid
  .m_axis_data_tdata(im2_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_re3 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_re3), // input s_axis_data_tvalid
  .s_axis_data_tready(tready1), // output s_axis_data_tready
  .s_axis_data_tdata(re3_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_re3), // output m_axis_data_tvalid
  .m_axis_data_tdata(re3_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_im3 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_im3), // input s_axis_data_tvalid
  .s_axis_data_tready(tready2), // output s_axis_data_tready
  .s_axis_data_tdata(im3_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_im3), // output m_axis_data_tvalid
  .m_axis_data_tdata(im3_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_re4 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_re4), // input s_axis_data_tvalid
  .s_axis_data_tready(tready1), // output s_axis_data_tready
  .s_axis_data_tdata(re4_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_re4), // output m_axis_data_tvalid
  .m_axis_data_tdata(re4_f2) // output [23 : 0] m_axis_data_tdata
);

fir fir_im4 (
  .aclk(clk), // input aclk
  .s_axis_data_tvalid(valid_im4), // input s_axis_data_tvalid
  .s_axis_data_tready(tready2), // output s_axis_data_tready
  .s_axis_data_tdata(im4_f1[15:0]+2), // input [15 : 0] s_axis_data_tdata
  .m_axis_data_tvalid(fvalid_im4), // output m_axis_data_tvalid
  .m_axis_data_tdata(im4_f2) // output [23 : 0] m_axis_data_tdata
);

//			******** short fifo before packing
fifoB fifo1_re1 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(re1_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_re1), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(re1_ff), // output [15 : 0]
  .full(full_2), // output full
  .empty(empty_re1) // output empty
);

fifoB fifo1_im1 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(im1_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_im1), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(im1_ff), // output [15 : 0]
  .full(full_1), // output full
  .empty(empty_im1) // output empty
);

fifoB fifo1_re2 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(re2_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_re2), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(re2_ff), // output [15 : 0]
  .full(full_2), // output full
  .empty(empty_re2) // output empty
);

fifoB fifo1_im2 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(im2_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_im2), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(im2_ff), // output [15 : 0]
  .full(full_1), // output full
  .empty(empty_im2) // output empty
);

fifoB fifo1_re3 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(re3_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_re3), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(re3_ff), // output [15 : 0]
  .full(full_2), // output full
  .empty(empty_re3) // output empty
);

fifoB fifo1_im3 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(im3_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_im3), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(im3_ff), // output [15 : 0]
  .full(full_1), // output full
  .empty(empty_im3) // output empty
);

fifoB fifo1_re4 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(re4_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_re4), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(re4_ff), // output [15 : 0]
  .full(full_2), // output full
  .empty(empty_re4) // output empty
);

fifoB fifo1_im4 (
  .clk(clk), // input clk
  .rst(rst), // input rst
  .din(im4_f2[23:7]), // input [15 : 0] din da gre DC skozi brez izgube
  .wr_en(fvalid_im4), // input wr_en
  .rd_en(rd_en), // input rd_en
  .dout(im4_ff), // output [15 : 0]
  .full(full_1), // output full
  .empty(empty_im4) // output empty
);

//				********** multiplexing and packing
packer2x8 pack1(
    .clk(clk),
    .rst(rst),
    .indata1(re1_ff),
    .indata2(im1_ff),
    .indata3(re2_ff),
    .indata4(im2_ff),
    .indata5(re3_ff),
    .indata6(im3_ff),
    .indata7(re4_ff),
    .indata8(im4_ff),
    .in_data_av(~empty_re1),	//PAZI!!! kaj ce fifota nista istocasno???
    .len(16'd4099),			//4099 = 4096 + 3 meta
    .pktyp(pktyp),
    .meta1(16'h2222),
    .meta2(16'h3333),
	 .rd_en(rd_en),
    .outdata(dout),
    .out_data_av(pk_data_av)
    );
	 
//				********** long fifo after packing
fifoA fifo2 (
  .rst(rst), 					// input rst
  .wr_clk(clk), 				// input wr_clk
  .rd_clk(ftdi_clk), 		// input rd_clk
  .din(doutr), 				// input [15 : 0] din direktno iz fifo1 ali packerja
  .wr_en(pk_data_av), 		// input wr_en  iz packerja
  .rd_en(fiforead),  		// input rd_en
  .dout(fifo_out), 			// output [7 : 0] dout
  .full(full_2), 				// output full
  .overflow(overflow), 		// output overflow
  .empty(fifoempty) 			// output empty
);

//state machine for USB read/write
localparam	WRITE=0,
				OEON=1,
				RDON=2,
				READ1=3,
				READ2=4,
				READ3=5,
				READ4=6,
				READ5=7,
				READ6=8,
				FLUSH=9;

reg [3:0] stat_d = WRITE;
reg [3:0] stat_q = WRITE;

always @(*)
	begin
	stat_d = stat_q;
	case(stat_q)
		WRITE:
			begin
			if (ftdi_rxf_n)
				begin
				stat_d = WRITE;
				end
			else
				begin
				stat_d = OEON;
				end
			fiforead = ~ftdi_txe_n;
			ftdi_data_r = fifo_out;
			ftdi_oe_n_r = 1;
			ftdi_rd_n_r = 1;
			ftdi_wr_n_r = fifoempty;
			commandstrobe = 0;
			end
		OEON:
			begin
			stat_d = READ1;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 1;
			ftdi_wr_n_r = 1;
			commandstrobe = 0;
			end
		RDON:
			begin
			stat_d = READ1;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandstrobe = 0;
			end
		READ1:
			begin
			stat_d = READ2;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[47:40] =  ftdi_data;
			commandstrobe = 0;
			end
		READ2:
			begin
			stat_d = READ3;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[39:32] =  ftdi_data;
			commandstrobe = 0;
			end
		READ3:
			begin
			stat_d = READ4;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[31:24] =  ftdi_data;
			commandstrobe = 0;
			end
		READ4:
			begin
			stat_d = FLUSH;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[23:16] =  ftdi_data;
			commandstrobe = 0;
			end
		READ5:
			begin
			stat_d = READ6;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[15:8] =  ftdi_data;
			commandstrobe = 0;
			end
		READ6:
			begin
			stat_d = FLUSH;
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandword[7:0] =  ftdi_data;
			commandstrobe = 0;
			end
		FLUSH:
			begin
			if (ftdi_rxf_n)
				begin
				stat_d = WRITE;
				end
			else
				begin
				stat_d = FLUSH;
				end
			fiforead = 0;
			ftdi_data_r = 8'hZZ;
			ftdi_oe_n_r = 0;
			ftdi_rd_n_r = 0;
			ftdi_wr_n_r = 1;
			commandstrobe = 1;
			end
	endcase
	end	//always @(*)
	
always @(posedge ftdi_clk)
	begin
	if (rst)
		begin
		stat_q <= WRITE;
		end
	else
		begin
		stat_q <= stat_d;
		end
	end

//			********** ftdi clock indicator (blinkenled)
ledblink blinkftdi(
	.clk(ftdi_clk),
	.rst(rst),
	.blink(zmig1)
	);
	
endmodule
